import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Film, User, Award, Play, Plus, Video, Scissors, Zap } from 'lucide-react'
import { useNavigate } from 'react-router-dom'
import ImageSlider from '../components/ImageSlider'
import './MyProjects.css'

const MyProjects = () => {
  const navigate = useNavigate()
  const [projects, setProjects] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('videoProjects')
    if (saved) {
      try {
        setProjects(JSON.parse(saved))
      } catch (e) {
        setProjects([])
      }
    } else {
      const defaultProjects = [
        {
          id: 1,
          course: 'Premiere Pro: Продвинутый уровень',
          instructor: 'Алексей Кинематографов',
          work: 'Короткометражный фильм "Городские ритмы"',
          rating: 5,
          photo: '/images/project-1.jpg'
        },
        {
          id: 2,
          course: 'DaVinci Resolve: Цветокоррекция',
          instructor: 'Мария Режиссерова',
          work: 'Музыкальный клип "Неоновые ночи"',
          rating: 4,
          photo: '/images/project-2.jpg'
        },
        {
          id: 3,
          course: 'Final Cut Pro: Монтаж для YouTube',
          instructor: 'Дмитрий Видеографов',
          work: 'Влог-серия "Путешествия"',
          rating: 5,
          photo: '/images/project-3.jpg'
        }
      ]
      setProjects(defaultProjects)
      localStorage.setItem('videoProjects', JSON.stringify(defaultProjects))
    }
  }, [])

  return (
    <div className="my-projects-film">
      <div className="film-strip-top"></div>
      
      <div className="container">
        <motion.section 
          className="hero-film-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.8 }}
        >
          <div className="hero-content-film">
            <motion.div 
              className="hero-badge-film"
              initial={{ scale: 0, rotate: -180 }}
              animate={{ scale: 1, rotate: 0 }}
              transition={{ delay: 0.2, type: "spring" }}
            >
              <Film size={18} strokeWidth={2.5} />
              <span>МОИ ПРОЕКТЫ</span>
            </motion.div>
            
            <motion.h1 
              className="hero-title-film"
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.3 }}
            >
              Мои <span className="title-accent-film">проекты</span>
            </motion.h1>
            
            <motion.p 
              className="hero-description-film"
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.4 }}
            >
              Отслеживайте свои работы, оценки преподавателей и продолжайте развиваться в видеомонтаже
            </motion.p>
            
            <motion.button 
              className="cta-film-button"
              onClick={() => navigate('/enroll')}
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.5 }}
              whileHover={{ scale: 1.05 }}
              whileTap={{ scale: 0.95 }}
            >
              <Plus size={22} strokeWidth={2.5} />
              <span>Новый курс</span>
            </motion.button>
          </div>
          
          <motion.div 
            className="hero-visual-film"
            initial={{ x: 100, opacity: 0 }}
            animate={{ x: 0, opacity: 1 }}
            transition={{ delay: 0.6 }}
          >
            <div className="film-reel">
              <div className="reel-hole"></div>
              <div className="reel-hole"></div>
              <div className="reel-hole"></div>
              <div className="reel-hole"></div>
              <div className="reel-center"></div>
            </div>
          </motion.div>
        </motion.section>

        <ImageSlider />

        <section className="projects-timeline-section">
          {projects.length === 0 ? (
            <motion.div 
              className="empty-state-film"
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
            >
              <div className="empty-icon-container-film">
                <Video size={64} strokeWidth={1.5} />
                <div className="empty-pulse-film"></div>
              </div>
              <h2 className="empty-title-film">Нет проектов</h2>
              <p className="empty-text-film">Начните с записи на первый курс</p>
              <button className="empty-button-film" onClick={() => navigate('/enroll')}>
                Записаться
              </button>
            </motion.div>
          ) : (
            <div className="projects-timeline-film">
              {projects.map((project, index) => {
                return (
                  <motion.div
                    key={project.id}
                    className="project-frame-film"
                    initial={{ opacity: 0, y: 50 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.2 + index * 0.15 }}
                    whileHover={{ y: -10 }}
                  >
                    <div className="frame-top-film">
                      <div className="frame-perforations">
                        {[...Array(8)].map((_, i) => (
                          <div key={i} className="perforation"></div>
                        ))}
                      </div>
                    </div>
                    
                    <div className="project-content-film">
                      <div className="project-image-film">
                        <img src={project.photo} alt={project.work} />
                        <div className="project-overlay-film">
                          <button className="play-button-film">
                            <Play size={32} strokeWidth={2.5} fill="currentColor" />
                          </button>
                        </div>
                        <div className="project-rating-film">
                          <Award size={18} strokeWidth={2} />
                          <span>{project.rating}/5</span>
                        </div>
                      </div>
                      
                      <div className="project-info-film">
                        <h3 className="project-title-film">{project.course}</h3>
                        <div className="project-meta-film">
                          <div className="meta-item-film">
                            <User size={18} strokeWidth={2} />
                            <span>{project.instructor}</span>
                          </div>
                          <div className="meta-item-film">
                            <Scissors size={18} strokeWidth={2} />
                            <span>{project.work}</span>
                          </div>
                        </div>
                      </div>
                    </div>
                    
                    <div className="frame-bottom-film">
                      <div className="frame-perforations">
                        {[...Array(8)].map((_, i) => (
                          <div key={i} className="perforation"></div>
                        ))}
                      </div>
                    </div>
                  </motion.div>
                )
              })}
            </div>
          )}
        </section>

        <section className="stats-film-section">
          <motion.div 
            className="stats-container-film"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
          >
            <div className="stat-item-film">
              <div className="stat-icon-film">
                <Film size={28} strokeWidth={2} />
              </div>
              <div className="stat-value-film">{projects.length}</div>
              <div className="stat-label-film">Проектов</div>
            </div>
            <div className="stat-item-film">
              <div className="stat-icon-film">
                <Award size={28} strokeWidth={2} />
              </div>
              <div className="stat-value-film">
                {projects.length > 0 
                  ? (projects.reduce((sum, p) => sum + (p.rating || 0), 0) / projects.length).toFixed(1)
                  : '0.0'}
              </div>
              <div className="stat-label-film">Средняя оценка</div>
            </div>
            <div className="stat-item-film">
              <div className="stat-icon-film">
                <Zap size={28} strokeWidth={2} />
              </div>
              <div className="stat-value-film">24/7</div>
              <div className="stat-label-film">Поддержка</div>
            </div>
          </motion.div>
        </section>
      </div>
      
      <div className="film-strip-bottom"></div>
    </div>
  )
}

export default MyProjects

